<?php

class Stripe {
    private $secret_key;
    private $publishable_key;
    private $api_url = 'https://api.stripe.com/v1';
    
    public function __construct($secret_key, $publishable_key = '') {
        $this->secret_key = $secret_key;
        $this->publishable_key = $publishable_key;
    }
    
    /**
     * Payment Intent oluştur
     */
    public function createPaymentIntent($amount, $currency = 'try', $metadata = []) {
        $amount_cents = (int)($amount * 100); // Stripe kuruş cinsinden çalışır
        
        $data = [
            'amount' => $amount_cents,
            'currency' => strtolower($currency),
            'metadata' => $metadata
        ];
        
        return $this->makeRequest('payment_intents', $data);
    }
    
    /**
     * Payment Intent detaylarını getir
     */
    public function retrievePaymentIntent($payment_intent_id) {
        return $this->makeRequest('payment_intents/' . $payment_intent_id, [], 'GET');
    }
    
    /**
     * Webhook event'i doğrula
     */
    public function verifyWebhook($payload, $signature, $webhook_secret) {
        $timestamp = '';
        $signatures = [];
        
        // Signature'dan timestamp ve signature'ları ayır
        $parts = explode(',', $signature);
        foreach ($parts as $part) {
            $kv = explode('=', $part, 2);
            if (count($kv) == 2) {
                if ($kv[0] == 't') {
                    $timestamp = $kv[1];
                } elseif ($kv[0] == 'v1') {
                    $signatures[] = $kv[1];
                }
            }
        }
        
        // Timestamp kontrolü (5 dakikadan eski ise reddet)
        if (time() - $timestamp > 300) {
            return false;
        }
        
        // Signature oluştur
        $signed_payload = $timestamp . '.' . $payload;
        $expected_signature = hash_hmac('sha256', $signed_payload, $webhook_secret);
        
        // Signature'ları karşılaştır
        foreach ($signatures as $signature) {
            if (hash_equals($expected_signature, $signature)) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * API isteği yap
     */
    private function makeRequest($endpoint, $data = [], $method = 'POST') {
        $url = $this->api_url . '/' . $endpoint;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_USERPWD, $this->secret_key . ':');
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/x-www-form-urlencoded'
        ]);
        
        if ($method == 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
        }
        
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            return [
                'error' => true,
                'message' => 'cURL Error: ' . $error
            ];
        }
        
        $result = json_decode($response, true);
        
        if ($http_code >= 200 && $http_code < 300) {
            return $result;
        } else {
            return [
                'error' => true,
                'message' => isset($result['error']['message']) ? $result['error']['message'] : 'Unknown error',
                'code' => $http_code,
                'raw' => $result
            ];
        }
    }
    
    /**
     * Publishable key'i döndür
     */
    public function getPublishableKey() {
        return $this->publishable_key;
    }
}
?>
