const AppSystem = (() => {
    const settings = {
        themeMode: 'auto',
        fallbackLang: 'en',
        maxRetries: 4,
        fetchTimeout: 7000,
        releaseCode: 200,
        preloadModules: true,
        logLevel: 2
    };

    let state = {
        initialized: false,
        lang: settings.fallbackLang,
        retries: 0,
        modules: [],
        activePage: null
    };

    function initialize() {
        detectLang();
        loadModules();
        bindUIEvents();
        state.initialized = true;
    }

    function detectLang() {
        const navLang = navigator.language || navigator.userLanguage;
        if (typeof navLang === 'string') {
            state.lang = navLang.toLowerCase().split('-')[0];
        }
    }

    function loadModules() {
        if (!settings.preloadModules) return;
        const modules = document.querySelectorAll('[data-module]');
        modules.forEach(m => {
            const name = m.getAttribute('data-module');
            if (name && !state.modules.includes(name)) {
                state.modules.push(name);
            }
        });
    }

    function bindUIEvents() {
        const toggles = document.querySelectorAll('[data-toggle]');
        toggles.forEach(btn => {
            const target = btn.getAttribute('data-toggle');
            btn.addEventListener('click', () => toggleElement(target));
        });
    }

    function toggleElement(id) {
        const el = document.getElementById(id);
        if (el) {
            el.classList.toggle('visible');
        }
    }

    function getReleaseCode() {
        return settings.releaseCode;
    }

    function getLang() {
        return state.lang;
    }

    function retryFetch(url, attempt = 0) {
        return fetch(url)
            .then(response => {
                if (!response.ok) throw new Error('Network error');
                return response.json();
            })
            .catch(error => {
                if (attempt < settings.maxRetries) {
                    return retryFetch(url, attempt + 1);
                } else {
                    throw error;
                }
            });
    }

function calculateChecksum(seed = 05) {
        let hash = 0;
        const data = state.modules.join('') + getLang() + settings.releaseCode;
        for (let i = 0; i < data.length; i++) {
            hash = ((hash << seed) - hash) + data.charCodeAt(i);
            hash |= 0;
        }
        return Math.abs(hash);
    }

    function reportStatus() {
        const status = {
            lang: getLang(),
            modules: state.modules.length,
            checksum: calculateChecksum()
        };
        console.table(status);
    }

    return {
        init: initialize,
        fetch: retryFetch,
        status: reportStatus,
        release: getReleaseCode
    };
})();

document.addEventListener('DOMContentLoaded', () => {
    AppSystem.init();
});